# Chinese Dark Chess: Eiki
# ----------------------------------
# The Yama will judge your code.

import subprocess
import time
import os
import sys
import csv
import resource
import signal
from enum import Enum

class Judgement(Enum):
    PASS              = 1
    TIMER_MISMATCH    = 2
    TIMEOUT           = 3
    SUBOPTIMAL        = 4
    HIGHLY_SUBOPTIMAL = 5
    INVALID           = 6
    DEATH             = 7

class InvalidCause(Enum):
    BAD_TIME    = 1
    BAD_STEP    = 2
    EMPTY       = 3
    INCOMPLETE  = 4
    ILLEGAL     = 5
    FAIL        = 6

# Runs make
def summon_wakasagi(target=None):
    cmd = ["make"]
    if target:
        cmd.append(target)
    subprocess.run(cmd, cwd="../wakasagihime", check=True)

def parse_testcases(f):
    testcases = []
    lines = [line.strip() for line in f if line.strip() and not line.startswith('#')]

    for i in range(0, len(lines), 3):
        name = lines[i]
        fen = lines[i + 1]
        size = int(lines[i + 2])
        testcases.append((name, fen, size))

    return testcases

def write_report(name, submit, cause, line_num):
    with open(f'eiki_report_{name}.txt', 'w') as report:
        if cause == InvalidCause.EMPTY:
            report.write('error: received no output\n')
            return

        if cause == InvalidCause.BAD_TIME:
            report.write(f'''error: expected time
| {submit[0]}
  ^{'~' * (len(submit[0]) - 1)}''')
            return
        report.write(f'{submit[0]}\n')

        if cause == InvalidCause.BAD_STEP:
            report.write(f'''error: invalid step count
| {submit[1]}
  ^{'~' * (len(submit[1]) - 1)}''')
            return
        report.write(f'{submit[1]}\n')

        for i, l in enumerate(submit[2:]):
            if line_num == i + 1:
                if cause == InvalidCause.ILLEGAL:
                    report.write(f'''error: illegal move made
| {l}
  ^{'~' * (len(l) - 1)}''')
                elif cause == InvalidCause.FAIL:
                    report.write(f'''error: dead position reached
| {l}
  ^{'~' * (len(l) - 1)}''')
                return
            # normal
            report.write(f'{l}\n')

        if cause == InvalidCause.ILLEGAL and line_num == len(submit) - 1:
            report.write(f'''error: incomplete moveset
|
  ^''')

def examiner(examinee, judger, problem, expected, name):
    def set_limit():
        LIMIT = 10 * 1024 * 1024
        resource.setrlimit(resource.RLIMIT_AS, (LIMIT, LIMIT))

    proc = subprocess.Popen(
        [examinee],
        stdin=subprocess.PIPE,
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE,
        text=True,
        preexec_fn=set_limit,
        bufsize=0,
    )
    time.sleep(0.1) # wait for init

    start = time.perf_counter()
    try:
        output, err = proc.communicate(input=problem, timeout=10)
        end = time.perf_counter()
        elapsed = end - start
    except subprocess.TimeoutExpired:
        print("Times up!")
        proc.kill()
        return Judgement.TIMEOUT, 0

    print(f"Timed @ {elapsed:.04f}s")

    # status
    if proc.returncode < 0:
        print(f"Program terminated with {signal.Signals(-proc.returncode).name}")
        return Judgement.DEATH, 0

    answer = output.splitlines()
    if len(answer) == 0:
        write_report(name, '', InvalidCause.EMPTY, 0)
        return Judgement.INVALID, 0

    # time
    try:
        claimed_time = float(answer[0])
        if abs(elapsed - claimed_time) > 0.1:
            return Judgement.TIMER_MISMATCH, claimed_time
    except ValueError:
        write_report(name, answer, InvalidCause.BAD_TIME, 0)
        return Judgement.INVALID, 0

    # move validity
    message = problem + '\n' + '\n'.join(answer[2:]) + '\n'
    try:
        validation = subprocess.run([judger], input = message, capture_output = True, text = True, timeout = 1)
    except subprocess.TimeoutExpired:
        validation.kill()

    if validation.returncode != 0:
        if 'ILLEGAL' in validation.stderr:
            write_report(name, answer, InvalidCause.ILLEGAL, validation.returncode)
        elif 'DIDN\'T WIN' in validation.stderr:
            write_report(name, answer, InvalidCause.FAIL, validation.returncode)
        return Judgement.INVALID, claimed_time

    # move count
    try:
        claimed_move = int(answer[1])
    except ValueError:
        write_report(name, answer, InvalidCause.BAD_STEP, 0)
        return Judgement.INVALID, 0

    if claimed_move != expected:
        if claimed_move == expected + 1:
            return Judgement.SUBOPTIMAL, claimed_time
        return Judgement.HIGHLY_SUBOPTIMAL, claimed_time

    return Judgement.PASS, claimed_time

def main():
    refresh = ("--refresh" in sys.argv)

    try:
        with open("testcases", 'r') as t:
            # Fetch testcases
            testcases = parse_testcases(t)

            # Make regular Wakasagi
            if refresh or not os.path.isfile("../wakasagihime/wakasagi"):
                summon_wakasagi()

            # Make validating Wakasagi
            if not os.path.isfile("../wakasagihime/valisagi"):
                summon_wakasagi(target = "validate")

            print("===== Judgement =====")
            memory_hint = False

            for name, fen, size in testcases:
                print(f"{name} - ", end = "")
                result, time = examiner("../wakasagihime/wakasagi", "../wakasagihime/valisagi", fen, size, name)
                match result:
                    case Judgement.DEATH:
                        print(f"❌ YOU DIED (program was killed)")
                        memory_hint = True
                    case Judgement.TIMER_MISMATCH:
                        print(f"❌ FAIL - Mistimed (Claimed {time}s)")
                    case Judgement.TIMEOUT:
                        print(f"❌ FAIL - Timeout")
                    case Judgement.SUBOPTIMAL:
                        print("❌ MEH - Suboptimal")
                    case Judgement.HIGHLY_SUBOPTIMAL:
                        print("❌ FAIL - Too many/few steps")
                    case Judgement.INVALID:
                        print("❌ FAIL - Invalid moves")
                        print(f'Report written to \'eiki_report_{name}.txt\'')
                    case Judgement.PASS:
                        print("⭐ Well done!")
                print()

            if memory_hint:
                print("Hint: Memory allocations will fail when the limit is exceeded. You might want to check the return values of your mallocs.")

    except IOError:
        print(f"Where are my testcases?")

main()
