/* main file for hw3
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef WITH_GBA
#include <gba_input.h>
#include <gba_timers.h>
#include <gba_video.h>
#include <gba_console.h>
#else
#include <mygba.h>
#endif

// input image
#include "input.h"
const u32 width = 240, height = 160, size = width*height;

extern "C" void boxfilter(u16* dst, const u16* src);
extern "C" void myfilter(u16* dst, const u16* src);

void setup_timer();
void start_timer();
void stop_timer();
u32 get_timer();
void init();
void setup_text_mode();
void setup_video_mode();
void draw_bitmap(const u16*);
void draw_text(const char*, const char*);
void setup_pad();
bool key_down();
bool key_up();
bool key_left();
bool key_right();
void debug_print(const char*);

int main(void)
{
	u16 *original = (u16*)input_pic;
	u16 *groundtruth = (u16*)malloc(sizeof(u16)*size);
	u16 *filtered = (u16*)malloc(sizeof(u16)*size);

	// groundtruth
	boxfilter(groundtruth, original);

	setup_timer();
	start_timer();

	u32 start = get_timer();
	myfilter(filtered, original);
	u32 end = get_timer();

	// verify correct or not
	bool correct = true;
	for(u32 y = 0; y < height; ++y)
		for(u32 x=0; x < width; ++x)
			if( groundtruth[width*y + x] != filtered[width*y + x]) {
				correct = false;
				break;
			}

    init();
	setup_video_mode();
	draw_bitmap(filtered);

	char result[256] = {0}, clocks[256] = {0};
	sprintf(result, "%s,", correct ? "CORRECT" : "WRONG");
	sprintf(clocks, "%u CLOCKS.", end-start);
	// debug_print(message);

	while (1) {
		setup_pad();
		if (key_up()) { // filtered image
			setup_video_mode();
			draw_bitmap(filtered);
		} else if (key_left()) { // original image
			setup_video_mode();
			draw_bitmap(original);
		} else if (key_right()) { // reference image
			setup_video_mode();
			draw_bitmap(groundtruth);
		} else if (key_down()) { // result and clocks
			setup_text_mode();
			draw_text(result, clocks);
		}
	}
	return 0;
}

void setup_timer()
{
#ifdef WITH_GBA
	// prepare the timer
	REG_TM0CNT_H &= 0xFFFC;
	REG_TM1CNT_H |= 0x0004;
	REG_TM0CNT_L = 0;
	REG_TM1CNT_L = 0;
#else
	M_TIM0CNT_SPEED_SELECT_SET(0);
	R_TIM1CNT |= 0x0004;
    REG_TM1D = 0;
    REG_TM0D = 0;
#endif
}

void start_timer()
{
#ifdef WITH_GBA
	REG_TM0CNT_H |= 0x0080;
	REG_TM1CNT_H |= 0x0080;
#else
	M_TIM0CNT_TIMER_START;
	M_TIM1CNT_TIMER_START;
#endif
}

void stop_timer()
{
#ifdef WITH_GBA
	REG_TM0CNT_H &= 0xFF7F;
	REG_TM1CNT_H &= 0xFF7F;
#else
    M_TIM0CNT_TIMER_STOP;
    M_TIM1CNT_TIMER_STOP;
#endif
}

u32 get_timer()
{
	u16 h, l;
#ifdef WITH_GBA
	l = REG_TM0CNT_L;
	h = REG_TM1CNT_L;
#else
	l = REG_TM0D;
	h = REG_TM1D;
#endif
	return (h<<16) + l;
}

void init()
{
#ifdef WITH_GBA
#else
    ham_Init();
#endif
}

void setup_text_mode()
{
#ifdef WITH_GBA
	consoleDemoInit();
#else
	ham_SetBgMode(0);
	ham_InitText(0);
#endif
}

void setup_video_mode()
{
#ifdef WITH_GBA
	SetMode(BG2_ON | MODE_3);
#else
    ham_SetBgMode(3);
#endif
}

void draw_bitmap(const u16 *bitmap)
{
#ifdef WITH_GBA
	memcpy((void*)VRAM, (void*)bitmap, sizeof(u16)*size);
#else
    ham_LoadBitmap(const_cast<u16*>(bitmap));
#endif
}

void draw_text(const char *result, const char *clocks)
{
#ifdef WITH_GBA
	iprintf("\x1b[8;0H");
	iprintf(result);
	iprintf("\x1b[10;0H");
	iprintf(clocks);
#else
	ham_DrawText(0, 8, const_cast<char*>(result));
	ham_DrawText(0, 10, const_cast<char*>(clocks));
#endif
}

#ifdef WITH_GBA
u16 key;
#endif

void setup_pad()
{
#ifdef WITH_GBA
	scanKeys();
	key = keysDown();
#else
	ham_UpdatePad();
#endif
}

bool key_down()
{
#ifdef WITH_GBA
	return key == KEY_DOWN;
#else
	return Pad.Pressed.Down;
#endif
}

bool key_up()
{
#ifdef WITH_GBA
	return key == KEY_UP;
#else
	return Pad.Pressed.Up;
#endif
}

bool key_left()
{
#ifdef WITH_GBA
	return key == KEY_LEFT;
#else
	return Pad.Pressed.Left;
#endif
}

bool key_right()
{
#ifdef WITH_GBA
	return key == KEY_RIGHT;
#else
	return Pad.Pressed.Right;
#endif
}

void debug_print(const char *s)
{
    asm volatile("mov r0, %0;"
                 "swi 0xff;"
                 : // no ouput
                 : "r" (s)
                 : "r0");
}

